import { useState, useEffect, useRef } from 'react';
import { Send, Bot, User, Loader2, Sparkles } from 'lucide-react';
import { supabase, type Project, type AIInteraction, type File } from '../lib/supabase';

interface AIChatProps {
  project: Project;
  onFilesGenerated?: () => void;
}

export function AIChat({ project, onFilesGenerated }: AIChatProps) {
  const [messages, setMessages] = useState<AIInteraction[]>([]);
  const [input, setInput] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    loadMessages();
  }, [project.id]);

  useEffect(() => {
    scrollToBottom();
  }, [messages]);

  async function loadMessages() {
    const { data, error } = await supabase
      .from('ai_interactions')
      .select('*')
      .eq('project_id', project.id)
      .order('created_at', { ascending: true });

    if (!error && data) {
      setMessages(data);
    }
  }

  async function sendMessage() {
    if (!input.trim() || isLoading) return;

    const userMessage = input.trim();
    setInput('');
    setIsLoading(true);

    const { data: userMsg } = await supabase
      .from('ai_interactions')
      .insert([{ project_id: project.id, role: 'user', content: userMessage }])
      .select()
      .single();

    if (userMsg) {
      setMessages([...messages, userMsg]);
    }

    const shouldGenerateCode = checkIfCodeGenerationNeeded(userMessage);

    if (shouldGenerateCode) {
      const generatedFiles = await generateCodeWithAI(userMessage);

      if (generatedFiles && generatedFiles.length > 0) {
        await createFilesInProject(generatedFiles);

        const fileList = generatedFiles.map(f => f.path).join(', ');
        const aiResponse = `I've generated the following files for you:\n\n${fileList}\n\nThe files have been created in your project. You can now view and edit them in the file tree!`;

        const { data: assistantMsg } = await supabase
          .from('ai_interactions')
          .insert([{ project_id: project.id, role: 'assistant', content: aiResponse }])
          .select()
          .single();

        if (assistantMsg) {
          setMessages(prev => [...prev, assistantMsg]);
        }

        if (onFilesGenerated) {
          onFilesGenerated();
        }
      } else {
        const aiResponse = generateAIResponse(userMessage);
        const { data: assistantMsg } = await supabase
          .from('ai_interactions')
          .insert([{ project_id: project.id, role: 'assistant', content: aiResponse }])
          .select()
          .single();

        if (assistantMsg) {
          setMessages(prev => [...prev, assistantMsg]);
        }
      }
    } else {
      const aiResponse = generateAIResponse(userMessage);
      const { data: assistantMsg } = await supabase
        .from('ai_interactions')
        .insert([{ project_id: project.id, role: 'assistant', content: aiResponse }])
        .select()
        .single();

      if (assistantMsg) {
        setMessages(prev => [...prev, assistantMsg]);
      }
    }

    setIsLoading(false);
  }

  function checkIfCodeGenerationNeeded(message: string): boolean {
    const lower = message.toLowerCase();
    const triggers = [
      'make', 'create', 'build', 'generate', 'write',
      'app', 'component', 'page', 'website'
    ];
    return triggers.some(trigger => lower.includes(trigger));
  }

  async function generateCodeWithAI(prompt: string): Promise<Array<{ path: string; content: string; language: string }> | null> {
    try {
      const apiUrl = `${import.meta.env.VITE_SUPABASE_URL}/functions/v1/generate-code`;

      const response = await fetch(apiUrl, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${import.meta.env.VITE_SUPABASE_ANON_KEY}`,
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ prompt, projectId: project.id }),
      });

      if (!response.ok) {
        throw new Error('Failed to generate code');
      }

      const data = await response.json();
      return data.files;
    } catch (error) {
      console.error('Error generating code:', error);
      return null;
    }
  }

  async function createFilesInProject(files: Array<{ path: string; content: string; language: string }>) {
    for (const file of files) {
      const { data: existingFile } = await supabase
        .from('files')
        .select('id')
        .eq('project_id', project.id)
        .eq('path', file.path)
        .maybeSingle();

      if (existingFile) {
        await supabase
          .from('files')
          .update({ content: file.content, language: file.language, updated_at: new Date().toISOString() })
          .eq('id', existingFile.id);
      } else {
        await supabase
          .from('files')
          .insert([{
            project_id: project.id,
            path: file.path,
            content: file.content,
            language: file.language,
          }]);
      }
    }
  }

  function generateAIResponse(userMessage: string): string {
    const lower = userMessage.toLowerCase();

    if (lower.includes('structure') || lower.includes('organize')) {
      return "For a well-organized project, I recommend this structure:\n\n- /src/components/ - React components\n- /src/lib/ - Utility functions and libraries\n- /src/hooks/ - Custom React hooks\n- /src/types/ - TypeScript type definitions\n- /src/styles/ - CSS and styling files\n- /public/ - Static assets";
    }

    if (lower.includes('best practice') || lower.includes('convention')) {
      return "Here are some coding best practices:\n\n1. Use meaningful variable and function names\n2. Keep functions small and focused (single responsibility)\n3. Write comments for complex logic\n4. Use TypeScript for type safety\n5. Organize imports at the top\n6. Handle errors gracefully\n7. Keep components modular and reusable";
    }

    if (lower.includes('typescript') || lower.includes('type')) {
      return "TypeScript provides type safety and better IDE support. Key practices:\n\n1. Define interfaces for props and data structures\n2. Use 'type' for unions and primitives\n3. Avoid 'any' when possible\n4. Use generics for reusable components\n5. Enable strict mode in tsconfig.json";
    }

    if (lower.includes('help') || lower.includes('what can you')) {
      return "I'm here to help with your project! I can:\n\n- Generate complete apps and components\n- Create working code for various features\n- Provide coding best practices\n- Help with TypeScript and React patterns\n- Assist with debugging strategies\n\nJust tell me what you want to build and I'll generate the code for you!";
    }

    if (lower.includes('error') || lower.includes('bug') || lower.includes('debug')) {
      return "When debugging:\n\n1. Check the browser console for error messages\n2. Verify your imports are correct\n3. Ensure all dependencies are installed\n4. Check for typos in variable/function names\n5. Use console.log() to track values\n6. Verify file paths are correct\n7. Check that components are properly exported/imported";
    }

    if (lower.includes('react') || lower.includes('component')) {
      return "React component tips:\n\n1. Use functional components with hooks\n2. Keep components small and focused\n3. Extract reusable logic into custom hooks\n4. Use props for component communication\n5. Implement proper key props in lists\n6. Use useState for local state\n7. Use useEffect for side effects";
    }

    return "I'm an AI code generator! Just tell me what you want to build and I'll create the code for you. For example:\n\n- 'Make a world clock app showing time in all timezones'\n- 'Create a todo list app'\n- 'Build a counter app'\n\nWhat would you like me to create?";
  }

  function scrollToBottom() {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  }

  function handleKeyPress(e: React.KeyboardEvent) {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  }

  return (
    <div className="w-96 bg-slate-800 border-l border-slate-700 flex flex-col">
      <div className="p-4 border-b border-slate-700">
        <div className="flex items-center gap-2">
          <Bot size={20} className="text-blue-400" />
          <h2 className="text-sm font-semibold text-slate-200">AI Assistant</h2>
        </div>
      </div>

      <div className="flex-1 overflow-y-auto p-4 space-y-4">
        {messages.length === 0 && (
          <div className="text-center text-slate-500 py-8">
            <Sparkles size={48} className="mx-auto mb-3 opacity-50 text-blue-400" />
            <p className="text-sm font-semibold text-slate-300">AI Code Generator</p>
            <p className="text-xs mt-2">Tell me what to build and I'll create the code!</p>
            <p className="text-xs mt-1 text-slate-600">Try: "Make a world clock app"</p>
          </div>
        )}

        {messages.map((msg) => (
          <div key={msg.id} className={`flex gap-3 ${msg.role === 'assistant' ? 'items-start' : 'items-start'}`}>
            <div className={`flex-shrink-0 w-8 h-8 rounded-full flex items-center justify-center ${
              msg.role === 'user' ? 'bg-blue-600' : 'bg-slate-700'
            }`}>
              {msg.role === 'user' ? <User size={16} /> : <Bot size={16} />}
            </div>
            <div className="flex-1 min-w-0">
              <div className={`rounded-lg p-3 ${
                msg.role === 'user' ? 'bg-blue-600/20 text-slate-100' : 'bg-slate-700/50 text-slate-200'
              }`}>
                <p className="text-sm whitespace-pre-wrap break-words">{msg.content}</p>
              </div>
            </div>
          </div>
        ))}

        {isLoading && (
          <div className="flex gap-3 items-start">
            <div className="flex-shrink-0 w-8 h-8 rounded-full bg-slate-700 flex items-center justify-center">
              <Bot size={16} />
            </div>
            <div className="flex-1">
              <div className="rounded-lg p-3 bg-slate-700/50">
                <Loader2 size={16} className="animate-spin text-slate-400" />
              </div>
            </div>
          </div>
        )}

        <div ref={messagesEndRef} />
      </div>

      <div className="p-4 border-t border-slate-700">
        <div className="flex gap-2">
          <textarea
            value={input}
            onChange={(e) => setInput(e.target.value)}
            onKeyPress={handleKeyPress}
            placeholder="Ask me anything..."
            className="flex-1 px-3 py-2 bg-slate-700 text-slate-100 rounded-lg text-sm focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none"
            rows={2}
            disabled={isLoading}
          />
          <button
            onClick={sendMessage}
            disabled={!input.trim() || isLoading}
            className="self-end px-4 py-2 bg-blue-600 hover:bg-blue-700 disabled:bg-slate-700 disabled:text-slate-500 text-white rounded-lg transition-colors"
          >
            <Send size={18} />
          </button>
        </div>
      </div>
    </div>
  );
}
