import { useState, useEffect } from 'react';
import { Save, RotateCcw } from 'lucide-react';
import { supabase, type File } from '../lib/supabase';

interface CodeEditorProps {
  file: File;
  onSave: () => void;
}

export function CodeEditor({ file, onSave }: CodeEditorProps) {
  const [content, setContent] = useState(file.content);
  const [hasChanges, setHasChanges] = useState(false);
  const [saving, setSaving] = useState(false);

  useEffect(() => {
    setContent(file.content);
    setHasChanges(false);
  }, [file.id]);

  function handleChange(value: string) {
    setContent(value);
    setHasChanges(value !== file.content);
  }

  async function saveFile() {
    setSaving(true);
    const { error } = await supabase
      .from('files')
      .update({ content, updated_at: new Date().toISOString() })
      .eq('id', file.id);

    if (!error) {
      setHasChanges(false);
      onSave();
    }
    setSaving(false);
  }

  function revertChanges() {
    setContent(file.content);
    setHasChanges(false);
  }

  return (
    <div className="flex-1 flex flex-col bg-slate-900">
      <div className="flex items-center justify-between px-4 py-2 bg-slate-800 border-b border-slate-700">
        <div className="flex items-center gap-3">
          <h3 className="text-sm font-medium text-slate-200">{file.path}</h3>
          {hasChanges && (
            <span className="text-xs text-amber-400">• Unsaved changes</span>
          )}
        </div>
        <div className="flex gap-2">
          {hasChanges && (
            <button
              onClick={revertChanges}
              className="flex items-center gap-1.5 px-3 py-1.5 bg-slate-700 hover:bg-slate-600 text-slate-200 rounded text-sm transition-colors"
            >
              <RotateCcw size={14} />
              Revert
            </button>
          )}
          <button
            onClick={saveFile}
            disabled={!hasChanges || saving}
            className="flex items-center gap-1.5 px-3 py-1.5 bg-blue-600 hover:bg-blue-700 disabled:bg-slate-700 disabled:text-slate-500 text-white rounded text-sm transition-colors"
          >
            <Save size={14} />
            {saving ? 'Saving...' : 'Save'}
          </button>
        </div>
      </div>

      <div className="flex-1 overflow-hidden">
        <textarea
          value={content}
          onChange={(e) => handleChange(e.target.value)}
          className="w-full h-full p-4 bg-slate-900 text-slate-100 font-mono text-sm focus:outline-none resize-none"
          spellCheck={false}
          style={{
            tabSize: 2,
            lineHeight: 1.6,
          }}
        />
      </div>

      <div className="px-4 py-2 bg-slate-800 border-t border-slate-700 text-xs text-slate-400">
        {file.language} • {content.split('\n').length} lines • {content.length} characters
      </div>
    </div>
  );
}
