import { useState, useEffect } from 'react';
import { File as FileIcon, Folder, ChevronRight, ChevronDown, Plus, Trash2, Download } from 'lucide-react';
import { supabase, type File, type Project } from '../lib/supabase';

interface FileTreeProps {
  project: Project;
  selectedFile: File | null;
  onSelectFile: (file: File) => void;
  onFilesChange: () => void;
}

type FileNode = {
  name: string;
  path: string;
  isDirectory: boolean;
  children?: FileNode[];
  file?: File;
};

export function FileTree({ project, selectedFile, onSelectFile, onFilesChange }: FileTreeProps) {
  const [files, setFiles] = useState<File[]>([]);
  const [fileTree, setFileTree] = useState<FileNode[]>([]);
  const [expandedPaths, setExpandedPaths] = useState<Set<string>>(new Set(['/']));
  const [showNewFile, setShowNewFile] = useState(false);
  const [newFilePath, setNewFilePath] = useState('');

  useEffect(() => {
    loadFiles();
  }, [project.id]);

  useEffect(() => {
    buildFileTree(files);
  }, [files]);

  async function loadFiles() {
    const { data, error } = await supabase
      .from('files')
      .select('*')
      .eq('project_id', project.id)
      .order('path');

    if (!error && data) {
      setFiles(data);
    }
  }

  function buildFileTree(fileList: File[]) {
    const root: FileNode[] = [];
    const pathMap = new Map<string, FileNode>();

    fileList.forEach((file) => {
      const parts = file.path.split('/').filter(Boolean);
      let currentPath = '';

      parts.forEach((part, index) => {
        const parentPath = currentPath;
        currentPath = currentPath + '/' + part;

        if (!pathMap.has(currentPath)) {
          const node: FileNode = {
            name: part,
            path: currentPath,
            isDirectory: index < parts.length - 1,
            children: index < parts.length - 1 ? [] : undefined,
            file: index === parts.length - 1 ? file : undefined,
          };

          pathMap.set(currentPath, node);

          if (parentPath === '') {
            root.push(node);
          } else {
            const parent = pathMap.get(parentPath);
            if (parent && parent.children) {
              parent.children.push(node);
            }
          }
        }
      });
    });

    setFileTree(root);
  }

  async function createFile() {
    if (!newFilePath.trim() || newFilePath.startsWith('/')) return;

    const path = '/' + newFilePath.trim();
    const language = getLanguageFromPath(path);

    const { data, error } = await supabase
      .from('files')
      .insert([{ project_id: project.id, path, content: '', language }])
      .select()
      .single();

    if (!error && data) {
      setFiles([...files, data]);
      setShowNewFile(false);
      setNewFilePath('');
      onFilesChange();
      onSelectFile(data);

      const dirPath = path.substring(0, path.lastIndexOf('/'));
      if (dirPath) {
        setExpandedPaths(new Set([...expandedPaths, dirPath]));
      }
    }
  }

  async function deleteFile(file: File, e: React.MouseEvent) {
    e.stopPropagation();
    if (!confirm(`Delete ${file.path}?`)) return;

    const { error } = await supabase.from('files').delete().eq('id', file.id);

    if (!error) {
      setFiles(files.filter(f => f.id !== file.id));
      onFilesChange();
      if (selectedFile?.id === file.id) {
        onSelectFile(null as any);
      }
    }
  }

  function getLanguageFromPath(path: string): string {
    const ext = path.split('.').pop()?.toLowerCase();
    const langMap: Record<string, string> = {
      js: 'javascript',
      jsx: 'javascript',
      ts: 'typescript',
      tsx: 'typescript',
      py: 'python',
      html: 'html',
      css: 'css',
      json: 'json',
      md: 'markdown',
      sql: 'sql',
      sh: 'shell',
      yml: 'yaml',
      yaml: 'yaml',
    };
    return langMap[ext || ''] || 'plaintext';
  }

  function toggleExpanded(path: string) {
    const newExpanded = new Set(expandedPaths);
    if (newExpanded.has(path)) {
      newExpanded.delete(path);
    } else {
      newExpanded.add(path);
    }
    setExpandedPaths(newExpanded);
  }

  async function downloadProject() {
    const blob = await createProjectZip();
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${project.name}.zip`;
    a.click();
    URL.revokeObjectURL(url);
  }

  async function createProjectZip(): Promise<Blob> {
    const JSZip = (await import('jszip')).default;
    const zip = new JSZip();

    files.forEach(file => {
      zip.file(file.path.substring(1), file.content);
    });

    return await zip.generateAsync({ type: 'blob' });
  }

  function renderNode(node: FileNode, level: number = 0) {
    const isExpanded = expandedPaths.has(node.path);
    const isSelected = selectedFile?.id === node.file?.id;

    return (
      <div key={node.path}>
        <div
          onClick={() => {
            if (node.isDirectory) {
              toggleExpanded(node.path);
            } else if (node.file) {
              onSelectFile(node.file);
            }
          }}
          className={`flex items-center gap-2 px-3 py-1.5 hover:bg-slate-700/50 cursor-pointer transition-colors group ${
            isSelected ? 'bg-slate-700' : ''
          }`}
          style={{ paddingLeft: `${level * 16 + 12}px` }}
        >
          {node.isDirectory ? (
            <>
              {isExpanded ? <ChevronDown size={16} className="text-slate-400" /> : <ChevronRight size={16} className="text-slate-400" />}
              <Folder size={16} className="text-blue-400" />
            </>
          ) : (
            <>
              <div style={{ width: 16 }} />
              <FileIcon size={16} className="text-slate-400" />
            </>
          )}
          <span className="text-sm text-slate-200 flex-1">{node.name}</span>
          {!node.isDirectory && node.file && (
            <button
              onClick={(e) => deleteFile(node.file!, e)}
              className="opacity-0 group-hover:opacity-100 p-1 hover:bg-red-600/20 rounded transition-all"
            >
              <Trash2 size={14} className="text-red-400" />
            </button>
          )}
        </div>
        {node.isDirectory && isExpanded && node.children && (
          <div>
            {node.children.map(child => renderNode(child, level + 1))}
          </div>
        )}
      </div>
    );
  }

  return (
    <div className="w-64 bg-slate-800 border-r border-slate-700 flex flex-col">
      <div className="p-3 border-b border-slate-700 space-y-2">
        <h2 className="text-sm font-semibold text-slate-300 truncate">{project.name}</h2>
        <div className="flex gap-1">
          <button
            onClick={() => setShowNewFile(!showNewFile)}
            className="flex-1 flex items-center justify-center gap-1.5 px-3 py-1.5 bg-slate-700 hover:bg-slate-600 text-slate-200 rounded text-sm transition-colors"
          >
            <Plus size={16} />
            New File
          </button>
          <button
            onClick={downloadProject}
            className="px-3 py-1.5 bg-slate-700 hover:bg-slate-600 text-slate-200 rounded transition-colors"
            title="Download project as ZIP"
          >
            <Download size={16} />
          </button>
        </div>
      </div>

      {showNewFile && (
        <div className="p-3 border-b border-slate-700 bg-slate-700/30">
          <input
            type="text"
            placeholder="path/to/file.js"
            value={newFilePath}
            onChange={(e) => setNewFilePath(e.target.value)}
            className="w-full px-2 py-1.5 bg-slate-900 text-slate-100 rounded text-sm focus:outline-none focus:ring-2 focus:ring-blue-500 mb-2"
            autoFocus
          />
          <div className="flex gap-1">
            <button
              onClick={createFile}
              className="flex-1 px-2 py-1 bg-green-600 hover:bg-green-700 text-white rounded text-xs transition-colors"
            >
              Create
            </button>
            <button
              onClick={() => {
                setShowNewFile(false);
                setNewFilePath('');
              }}
              className="flex-1 px-2 py-1 bg-slate-600 hover:bg-slate-500 text-slate-200 rounded text-xs transition-colors"
            >
              Cancel
            </button>
          </div>
        </div>
      )}

      <div className="flex-1 overflow-y-auto">
        {fileTree.map(node => renderNode(node))}
        {files.length === 0 && !showNewFile && (
          <div className="p-6 text-center text-slate-500">
            <FileIcon size={32} className="mx-auto mb-2 opacity-50" />
            <p className="text-sm">No files yet</p>
          </div>
        )}
      </div>
    </div>
  );
}
