import { useState, useEffect } from 'react';
import { Plus, FolderOpen, Trash2, Calendar } from 'lucide-react';
import { supabase, type Project } from '../lib/supabase';

interface ProjectListProps {
  onSelectProject: (project: Project) => void;
}

export function ProjectList({ onSelectProject }: ProjectListProps) {
  const [projects, setProjects] = useState<Project[]>([]);
  const [showNewProject, setShowNewProject] = useState(false);
  const [newProjectName, setNewProjectName] = useState('');
  const [newProjectDesc, setNewProjectDesc] = useState('');

  useEffect(() => {
    loadProjects();
  }, []);

  async function loadProjects() {
    const { data, error } = await supabase
      .from('projects')
      .select('*')
      .order('updated_at', { ascending: false });

    if (!error && data) {
      setProjects(data);
    }
  }

  async function createProject() {
    if (!newProjectName.trim()) return;

    const { data, error } = await supabase
      .from('projects')
      .insert([{ name: newProjectName, description: newProjectDesc }])
      .select()
      .single();

    if (!error && data) {
      setProjects([data, ...projects]);
      setShowNewProject(false);
      setNewProjectName('');
      setNewProjectDesc('');
      onSelectProject(data);
    }
  }

  async function deleteProject(id: string, e: React.MouseEvent) {
    e.stopPropagation();
    if (!confirm('Delete this project and all its files?')) return;

    const { error } = await supabase.from('projects').delete().eq('id', id);

    if (!error) {
      setProjects(projects.filter(p => p.id !== id));
    }
  }

  return (
    <div className="w-80 bg-slate-900 border-r border-slate-800 flex flex-col">
      <div className="p-4 border-b border-slate-800">
        <h1 className="text-xl font-bold text-slate-100 mb-4">AI Project Manager</h1>
        <button
          onClick={() => setShowNewProject(!showNewProject)}
          className="w-full flex items-center justify-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg transition-colors"
        >
          <Plus size={20} />
          New Project
        </button>
      </div>

      {showNewProject && (
        <div className="p-4 border-b border-slate-800 bg-slate-800/50">
          <input
            type="text"
            placeholder="Project name"
            value={newProjectName}
            onChange={(e) => setNewProjectName(e.target.value)}
            className="w-full px-3 py-2 bg-slate-700 text-slate-100 rounded-lg mb-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            autoFocus
          />
          <textarea
            placeholder="Description (optional)"
            value={newProjectDesc}
            onChange={(e) => setNewProjectDesc(e.target.value)}
            className="w-full px-3 py-2 bg-slate-700 text-slate-100 rounded-lg mb-2 focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none"
            rows={2}
          />
          <div className="flex gap-2">
            <button
              onClick={createProject}
              className="flex-1 px-3 py-1.5 bg-green-600 hover:bg-green-700 text-white rounded text-sm transition-colors"
            >
              Create
            </button>
            <button
              onClick={() => {
                setShowNewProject(false);
                setNewProjectName('');
                setNewProjectDesc('');
              }}
              className="flex-1 px-3 py-1.5 bg-slate-700 hover:bg-slate-600 text-slate-300 rounded text-sm transition-colors"
            >
              Cancel
            </button>
          </div>
        </div>
      )}

      <div className="flex-1 overflow-y-auto">
        {projects.map((project) => (
          <div
            key={project.id}
            onClick={() => onSelectProject(project)}
            className="p-4 border-b border-slate-800 hover:bg-slate-800/50 cursor-pointer transition-colors group"
          >
            <div className="flex items-start justify-between mb-1">
              <div className="flex items-center gap-2 flex-1 min-w-0">
                <FolderOpen size={18} className="text-blue-400 flex-shrink-0" />
                <h3 className="font-medium text-slate-100 truncate">{project.name}</h3>
              </div>
              <button
                onClick={(e) => deleteProject(project.id, e)}
                className="opacity-0 group-hover:opacity-100 p-1 hover:bg-red-600/20 rounded transition-all"
              >
                <Trash2 size={16} className="text-red-400" />
              </button>
            </div>
            {project.description && (
              <p className="text-sm text-slate-400 mb-2 line-clamp-2">{project.description}</p>
            )}
            <div className="flex items-center gap-1 text-xs text-slate-500">
              <Calendar size={12} />
              {new Date(project.updated_at).toLocaleDateString()}
            </div>
          </div>
        ))}

        {projects.length === 0 && !showNewProject && (
          <div className="p-8 text-center text-slate-500">
            <FolderOpen size={48} className="mx-auto mb-3 opacity-50" />
            <p>No projects yet</p>
            <p className="text-sm mt-1">Create your first project to get started</p>
          </div>
        )}
      </div>
    </div>
  );
}
