import "jsr:@supabase/functions-js/edge-runtime.d.ts";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Methods": "GET, POST, PUT, DELETE, OPTIONS",
  "Access-Control-Allow-Headers": "Content-Type, Authorization, X-Client-Info, Apikey",
};

Deno.serve(async (req: Request) => {
  if (req.method === "OPTIONS") {
    return new Response(null, {
      status: 200,
      headers: corsHeaders,
    });
  }

  try {
    const { prompt, projectId } = await req.json();

    const generatedFiles = generateCode(prompt);

    return new Response(
      JSON.stringify({ files: generatedFiles }),
      {
        headers: {
          ...corsHeaders,
          'Content-Type': 'application/json',
        },
      }
    );
  } catch (error) {
    return new Response(
      JSON.stringify({ error: error.message }),
      {
        status: 500,
        headers: {
          ...corsHeaders,
          'Content-Type': 'application/json',
        },
      }
    );
  }
});

function generateCode(prompt: string): Array<{ path: string; content: string; language: string }> {
  const lower = prompt.toLowerCase();

  if (lower.includes('time') && lower.includes('zone')) {
    return [
      {
        path: '/src/components/WorldClock.tsx',
        language: 'typescript',
        content: `import { useState, useEffect } from 'react';
import { Clock } from 'lucide-react';

const timezones = [
  { name: 'New York', timezone: 'America/New_York' },
  { name: 'London', timezone: 'Europe/London' },
  { name: 'Tokyo', timezone: 'Asia/Tokyo' },
  { name: 'Sydney', timezone: 'Australia/Sydney' },
  { name: 'Dubai', timezone: 'Asia/Dubai' },
  { name: 'Los Angeles', timezone: 'America/Los_Angeles' },
  { name: 'Paris', timezone: 'Europe/Paris' },
  { name: 'Singapore', timezone: 'Asia/Singapore' },
  { name: 'Mumbai', timezone: 'Asia/Kolkata' },
  { name: 'Moscow', timezone: 'Europe/Moscow' },
];

export function WorldClock() {
  const [times, setTimes] = useState<Record<string, string>>({});

  useEffect(() => {
    const updateTimes = () => {
      const newTimes: Record<string, string> = {};
      timezones.forEach(({ name, timezone }) => {
        const time = new Date().toLocaleString('en-US', {
          timeZone: timezone,
          hour: '2-digit',
          minute: '2-digit',
          second: '2-digit',
          hour12: true,
        });
        const date = new Date().toLocaleString('en-US', {
          timeZone: timezone,
          weekday: 'short',
          month: 'short',
          day: 'numeric',
        });
        newTimes[name] = \`\${date} • \${time}\`;
      });
      setTimes(newTimes);
    };

    updateTimes();
    const interval = setInterval(updateTimes, 1000);

    return () => clearInterval(interval);
  }, []);

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-900 via-blue-900 to-slate-900 p-8">
      <div className="max-w-6xl mx-auto">
        <div className="text-center mb-12">
          <div className="flex items-center justify-center gap-3 mb-4">
            <Clock size={48} className="text-blue-400" />
            <h1 className="text-5xl font-bold text-white">World Clock</h1>
          </div>
          <p className="text-blue-200 text-lg">Live time across major cities worldwide</p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {timezones.map(({ name }) => (
            <div
              key={name}
              className="bg-white/10 backdrop-blur-sm border border-white/20 rounded-2xl p-6 hover:bg-white/15 transition-all duration-300 hover:scale-105"
            >
              <h2 className="text-2xl font-bold text-white mb-2">{name}</h2>
              <p className="text-blue-200 text-lg font-mono">
                {times[name] || 'Loading...'}
              </p>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
}`
      },
      {
        path: '/src/App.tsx',
        language: 'typescript',
        content: `import { WorldClock } from './components/WorldClock';

function App() {
  return <WorldClock />;
}

export default App;`
      }
    ];
  }

  if (lower.includes('todo') || lower.includes('task')) {
    return [
      {
        path: '/src/components/TodoApp.tsx',
        language: 'typescript',
        content: `import { useState } from 'react';
import { Plus, Trash2, Check } from 'lucide-react';

type Todo = {
  id: number;
  text: string;
  completed: boolean;
};

export function TodoApp() {
  const [todos, setTodos] = useState<Todo[]>([]);
  const [input, setInput] = useState('');

  function addTodo() {
    if (!input.trim()) return;
    setTodos([...todos, { id: Date.now(), text: input, completed: false }]);
    setInput('');
  }

  function toggleTodo(id: number) {
    setTodos(todos.map(todo => 
      todo.id === id ? { ...todo, completed: !todo.completed } : todo
    ));
  }

  function deleteTodo(id: number) {
    setTodos(todos.filter(todo => todo.id !== id));
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-900 to-blue-900 p-8">
      <div className="max-w-2xl mx-auto">
        <h1 className="text-4xl font-bold text-white mb-8 text-center">Todo List</h1>
        
        <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 mb-6">
          <div className="flex gap-2">
            <input
              type="text"
              value={input}
              onChange={(e) => setInput(e.target.value)}
              onKeyPress={(e) => e.key === 'Enter' && addTodo()}
              placeholder="Add a new task..."
              className="flex-1 px-4 py-3 rounded-lg bg-white/20 text-white placeholder-white/50 focus:outline-none focus:ring-2 focus:ring-blue-400"
            />
            <button
              onClick={addTodo}
              className="px-6 py-3 bg-blue-500 hover:bg-blue-600 text-white rounded-lg transition-colors"
            >
              <Plus size={24} />
            </button>
          </div>
        </div>

        <div className="space-y-3">
          {todos.map(todo => (
            <div
              key={todo.id}
              className="bg-white/10 backdrop-blur-sm rounded-xl p-4 flex items-center gap-3 hover:bg-white/15 transition-colors"
            >
              <button
                onClick={() => toggleTodo(todo.id)}
                className={\`w-6 h-6 rounded-full border-2 flex items-center justify-center transition-colors \${todo.completed ? 'bg-green-500 border-green-500' : 'border-white/50'}\`}
              >
                {todo.completed && <Check size={16} className="text-white" />}
              </button>
              <span className={\`flex-1 text-white \${todo.completed ? 'line-through opacity-50' : ''}\`}>
                {todo.text}
              </span>
              <button
                onClick={() => deleteTodo(todo.id)}
                className="p-2 hover:bg-red-500/20 rounded-lg transition-colors"
              >
                <Trash2 size={18} className="text-red-400" />
              </button>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
}`
      },
      {
        path: '/src/App.tsx',
        language: 'typescript',
        content: `import { TodoApp } from './components/TodoApp';

function App() {
  return <TodoApp />;
}

export default App;`
      }
    ];
  }

  if (lower.includes('counter')) {
    return [
      {
        path: '/src/components/Counter.tsx',
        language: 'typescript',
        content: `import { useState } from 'react';
import { Plus, Minus, RotateCcw } from 'lucide-react';

export function Counter() {
  const [count, setCount] = useState(0);

  return (
    <div className="min-h-screen bg-gradient-to-br from-green-900 to-teal-900 flex items-center justify-center">
      <div className="bg-white/10 backdrop-blur-sm rounded-3xl p-12 text-center">
        <h1 className="text-6xl font-bold text-white mb-8">Counter</h1>
        <div className="text-8xl font-bold text-white mb-12">{count}</div>
        <div className="flex gap-4">
          <button
            onClick={() => setCount(count - 1)}
            className="px-8 py-4 bg-red-500 hover:bg-red-600 text-white rounded-xl transition-colors"
          >
            <Minus size={32} />
          </button>
          <button
            onClick={() => setCount(0)}
            className="px-8 py-4 bg-gray-500 hover:bg-gray-600 text-white rounded-xl transition-colors"
          >
            <RotateCcw size={32} />
          </button>
          <button
            onClick={() => setCount(count + 1)}
            className="px-8 py-4 bg-green-500 hover:bg-green-600 text-white rounded-xl transition-colors"
          >
            <Plus size={32} />
          </button>
        </div>
      </div>
    </div>
  );
}`
      },
      {
        path: '/src/App.tsx',
        language: 'typescript',
        content: `import { Counter } from './components/Counter';

function App() {
  return <Counter />;
}

export default App;`
      }
    ];
  }

  return [
    {
      path: '/README.md',
      language: 'markdown',
      content: `# AI Generated Project\n\nI can generate code for various types of applications!\n\n## Try asking me to create:\n- A world clock showing time in all timezones\n- A todo list app\n- A counter app\n- And more!\n\nJust describe what you want to build and I'll generate the code for you.`
    }
  ];
}