/*
  # AI Project Management Platform Schema

  ## Overview
  This migration creates the database schema for an AI-powered project management 
  and coding platform with file management capabilities.

  ## New Tables

  ### 1. `projects`
  Stores project metadata and settings
  - `id` (uuid, primary key) - Unique project identifier
  - `name` (text) - Project name
  - `description` (text) - Project description
  - `created_at` (timestamptz) - Creation timestamp
  - `updated_at` (timestamptz) - Last update timestamp

  ### 2. `files`
  Stores project files and their content
  - `id` (uuid, primary key) - Unique file identifier
  - `project_id` (uuid, foreign key) - References projects table
  - `path` (text) - File path within project (e.g., /src/index.ts)
  - `content` (text) - File content
  - `language` (text) - Programming language for syntax highlighting
  - `created_at` (timestamptz) - Creation timestamp
  - `updated_at` (timestamptz) - Last update timestamp

  ### 3. `ai_interactions`
  Stores AI chat history and interactions per project
  - `id` (uuid, primary key) - Unique interaction identifier
  - `project_id` (uuid, foreign key) - References projects table
  - `role` (text) - Message role (user/assistant)
  - `content` (text) - Message content
  - `created_at` (timestamptz) - Creation timestamp

  ## Security
  - RLS enabled on all tables
  - Public access policies since no authentication is required
  - All users can perform all operations on all tables

  ## Notes
  1. No authentication system as per requirements
  2. File versioning can be added later if needed
  3. AI interactions are preserved for context
*/

-- Create projects table
CREATE TABLE IF NOT EXISTS projects (
  id uuid PRIMARY KEY DEFAULT gen_random_uuid(),
  name text NOT NULL,
  description text DEFAULT '',
  created_at timestamptz DEFAULT now(),
  updated_at timestamptz DEFAULT now()
);

-- Create files table
CREATE TABLE IF NOT EXISTS files (
  id uuid PRIMARY KEY DEFAULT gen_random_uuid(),
  project_id uuid NOT NULL REFERENCES projects(id) ON DELETE CASCADE,
  path text NOT NULL,
  content text DEFAULT '',
  language text DEFAULT 'plaintext',
  created_at timestamptz DEFAULT now(),
  updated_at timestamptz DEFAULT now(),
  UNIQUE(project_id, path)
);

-- Create AI interactions table
CREATE TABLE IF NOT EXISTS ai_interactions (
  id uuid PRIMARY KEY DEFAULT gen_random_uuid(),
  project_id uuid NOT NULL REFERENCES projects(id) ON DELETE CASCADE,
  role text NOT NULL CHECK (role IN ('user', 'assistant')),
  content text NOT NULL,
  created_at timestamptz DEFAULT now()
);

-- Create indexes
CREATE INDEX IF NOT EXISTS idx_files_project_id ON files(project_id);
CREATE INDEX IF NOT EXISTS idx_ai_interactions_project_id ON ai_interactions(project_id);
CREATE INDEX IF NOT EXISTS idx_ai_interactions_created_at ON ai_interactions(created_at);

-- Enable RLS
ALTER TABLE projects ENABLE ROW LEVEL SECURITY;
ALTER TABLE files ENABLE ROW LEVEL SECURITY;
ALTER TABLE ai_interactions ENABLE ROW LEVEL SECURITY;

-- Create policies for public access (no authentication required)
CREATE POLICY "Public can view all projects"
  ON projects FOR SELECT
  USING (true);

CREATE POLICY "Public can insert projects"
  ON projects FOR INSERT
  WITH CHECK (true);

CREATE POLICY "Public can update projects"
  ON projects FOR UPDATE
  USING (true)
  WITH CHECK (true);

CREATE POLICY "Public can delete projects"
  ON projects FOR DELETE
  USING (true);

CREATE POLICY "Public can view all files"
  ON files FOR SELECT
  USING (true);

CREATE POLICY "Public can insert files"
  ON files FOR INSERT
  WITH CHECK (true);

CREATE POLICY "Public can update files"
  ON files FOR UPDATE
  USING (true)
  WITH CHECK (true);

CREATE POLICY "Public can delete files"
  ON files FOR DELETE
  USING (true);

CREATE POLICY "Public can view all AI interactions"
  ON ai_interactions FOR SELECT
  USING (true);

CREATE POLICY "Public can insert AI interactions"
  ON ai_interactions FOR INSERT
  WITH CHECK (true);

CREATE POLICY "Public can update AI interactions"
  ON ai_interactions FOR UPDATE
  USING (true)
  WITH CHECK (true);

CREATE POLICY "Public can delete AI interactions"
  ON ai_interactions FOR DELETE
  USING (true);